document.addEventListener("DOMContentLoaded", () => {
  // Language switcher (front only)
  const langSwitcher = document.querySelector(".language-switcher");
  const langCurrent = document.querySelector(".lang-current");
  const langMenu = document.querySelector(".lang-menu");

  if (langSwitcher && langCurrent && langMenu) {
    langCurrent.addEventListener("click", () => {
      langSwitcher.classList.toggle("open");
    });

    langMenu.querySelectorAll("button").forEach((btn) => {
      btn.addEventListener("click", () => {
        const lang = (btn.dataset.lang || "").toUpperCase();
        if (lang) {
          langCurrent.textContent = lang + " ▾";
        }
        langSwitcher.classList.remove("open");
      });
    });

    document.addEventListener("click", (e) => {
      if (!langSwitcher.contains(e.target)) {
        langSwitcher.classList.remove("open");
      }
    });
  }

  // Burger / mobile nav
  const burger = document.getElementById("burger");
  const mobileNav = document.getElementById("mobileNav");

  function closeMobileNav() {
    if (burger) burger.classList.remove("open");
    if (mobileNav) mobileNav.classList.remove("open");
  }

  if (burger && mobileNav) {
    burger.addEventListener("click", () => {
      burger.classList.toggle("open");
      mobileNav.classList.toggle("open");
    });
  }

  // Views / routing
  const views = document.querySelectorAll(".view");

  function activateView(viewId) {
    if (!viewId) return;
    views.forEach((v) => {
      v.classList.toggle("view--active", v.id === viewId);
    });

    // Header nav active state
    document.querySelectorAll(".nav-link").forEach((link) => {
      const linkView = link.dataset.view;
      link.classList.toggle("nav-link--active", linkView === viewId);
    });

    // Mobile nav active state
    document.querySelectorAll(".mobile-nav-link").forEach((link) => {
      const linkView = link.dataset.view;
      link.classList.toggle("mobile-nav-link--active", linkView === viewId);
    });

    window.scrollTo({ top: 0, behavior: "smooth" });
  }

  // Generic click handler for anything with data-view
  document.querySelectorAll("[data-view]").forEach((el) => {
    el.addEventListener("click", (e) => {
      const viewId = el.dataset.view;
      if (!viewId) return;
      e.preventDefault();
      activateView(viewId);
      closeMobileNav();
    });
  });

  // Scroll helper buttons (top bar)
  document.querySelectorAll("[data-scroll]").forEach((el) => {
    el.addEventListener("click", (e) => {
      const targetId = el.dataset.scroll;
      if (!targetId) return;
      e.preventDefault();
      const target =
        document.getElementById(targetId + "Card") ||
        document.getElementById(targetId);
      if (target) {
        if (!document.getElementById("view-casino")?.classList.contains("view--active")) {
          activateView("view-casino");
        }
        target.scrollIntoView({ behavior: "smooth", block: "center" });
      }
      closeMobileNav();
    });
  });

  // Banner slider
  const slider = document.getElementById("bannerSlider");
  if (slider) {
    const slides = Array.from(slider.querySelectorAll(".banner-slide"));
    const paginationEl = document.getElementById("bannerPagination");

    let currentIndex = 0;

    function goTo(index) {
      slides.forEach((slide, i) => {
        slide.classList.toggle("banner-slide--active", i === index);
      });
      if (paginationEl) {
        const dots = Array.from(paginationEl.querySelectorAll(".banner-dot"));
        dots.forEach((dot, i) => {
          dot.classList.toggle("active", i === index);
        });
      }
      currentIndex = index;
    }

    if (slides.length > 0 && paginationEl) {
      slides.forEach((_s, i) => {
        const dot = document.createElement("button");
        dot.className = "banner-dot" + (i === 0 ? " active" : "");
        dot.type = "button";
        dot.addEventListener("click", () => goTo(i));
        paginationEl.appendChild(dot);
      });
    }

    let sliderInterval;
    function startSlider() {
      if (slides.length > 1) {
        sliderInterval = setInterval(() => {
          const next = (currentIndex + 1) % slides.length;
          goTo(next);
        }, 5000);
      }
    }
    function stopSlider() {
      if (sliderInterval) clearInterval(sliderInterval);
    }

    slider.addEventListener("mouseenter", stopSlider);
    slider.addEventListener("mouseleave", startSlider);

    goTo(0);
    startSlider();
  }

  // Providers
  const providersRowCompact = document.getElementById("providersRow");
  const providersRowFull = document.getElementById("providersRowFull");
  const providers = [
    "NeonSpin",
    "SkyForge",
    "CrashLab",
    "LiveCore",
    "Pyramid Labs",
    "MiniGames",
    "Red Tiger",
    "Pragmatic Play",
    "NetEnt",
    "Yggdrasil",
    "BGaming",
    "Hacksaw",
    "Quickspin",
    "Spinomenal",
    "Elk Studios",
    "NoLimit City",
  ];

  if (providersRowCompact) {
    providers.forEach((name, index) => {
      const pill = document.createElement("span");
      pill.className = "provider-pill" + (index < 4 ? " featured" : "");
      pill.textContent = name;
      providersRowCompact.appendChild(pill);
    });
  }

  if (providersRowFull) {
    providers.forEach((name, index) => {
      const pill = document.createElement("span");
      pill.className = "provider-pill" + (index < 6 ? " featured" : "");
      pill.textContent = name;
      providersRowFull.appendChild(pill);
    });
  }

  // Games category chips in casino view (only visual state now)
  const categoryTabs = document.getElementById("categoryTabs");
  if (categoryTabs) {
    categoryTabs.querySelectorAll(".chip").forEach((chip) => {
      chip.addEventListener("click", () => {
        categoryTabs.querySelectorAll(".chip").forEach((c) => c.classList.remove("chip--active"));
        chip.classList.add("chip--active");
      });
    });
  }

  // Scroll to top
  const scrollTopBtn = document.getElementById("scrollTopBtn");
  if (scrollTopBtn) {
    scrollTopBtn.addEventListener("click", () => {
      window.scrollTo({ top: 0, behavior: "smooth" });
    });
  }

  // Modals
  function openModal(kind) {
    if (kind === "support") {
      document.getElementById("supportModal")?.classList.add("open");
    } else if (kind === "auth") {
      document.getElementById("authModal")?.classList.add("open");
    } else if (kind === "cashier") {
      makeToast("Open your cashier / deposit module here.");
    }
  }

  function closeModal(kind) {
    if (kind === "support") {
      document.getElementById("supportModal")?.classList.remove("open");
    } else if (kind === "auth") {
      document.getElementById("authModal")?.classList.remove("open");
    }
  }

  document.querySelectorAll("[data-open]").forEach((el) => {
    el.addEventListener("click", () => {
      const kind = el.dataset.open;
      if (kind === "auth" && el.dataset.authMode) {
        setAuthMode(el.dataset.authMode);
      }
      openModal(kind);
    });
  });

  document.querySelectorAll("[data-close]").forEach((el) => {
    el.addEventListener("click", () => {
      const kind = el.dataset.close;
      closeModal(kind);
    });
  });

  // Close modals with Escape
  document.addEventListener("keydown", (e) => {
    if (e.key === "Escape") {
      closeModal("support");
      closeModal("auth");
    }
  });

  // Auth modal mode switching
  const authTitle = document.getElementById("authTitle");
  const authSubtitle = document.getElementById("authSubtitle");
  const authExtraRegister = document.getElementById("authExtraRegister");

  function setAuthMode(mode) {
    if (!authTitle || !authSubtitle || !authExtraRegister) return;
    const isRegister = mode === "register";
    authTitle.textContent = isRegister ? "Sign up" : "Log in";
    authSubtitle.textContent = isRegister
      ? "Create a new Telbet account. In a real project, connect this to your registration API."
      : "Enter your credentials to continue. This is a visual demo only.";
    authExtraRegister.hidden = !isRegister;
  }

  document.querySelectorAll("[data-auth-switch]").forEach((el) => {
    el.addEventListener("click", () => {
      const mode = el.dataset.authSwitch || "login";
      setAuthMode(mode);
    });
  });

  // Toast helper
  const toastContainer = document.getElementById("toastContainer");

  function makeToast(message) {
    if (!toastContainer) return;
    const div = document.createElement("div");
    div.className = "toast";
    div.textContent = message;
    toastContainer.appendChild(div);
    setTimeout(() => {
      div.style.opacity = "0";
      div.style.transition = "opacity 0.3s ease";
      setTimeout(() => div.remove(), 300);
    }, 1800);
  }

  // Top bar cashier quick menu (in header we call cashier modal via toast)
  document.querySelectorAll("[data-open='cashier']").forEach((btn) => {
    btn.addEventListener("click", () => {
      makeToast("Open your cashier / deposit module here.");
    });
  });

  // Initialise default view
  activateView("view-casino");
});
